package org.msh.reports.tableoperations;

import org.msh.reports.datatable.DataTable;
import org.msh.reports.datatable.Row;
import org.msh.reports.indicator.*;
import org.msh.reports.keys.Key;
import org.msh.reports.variables.Variable;

import java.util.ArrayList;
import java.util.List;

/**
 * Generates an indicator table from a data table and the variables
 * that generated the data table
 * @author Ricardo Memoria
 *
 */
public class IndicatorTransform {

    private List<Variable> varRows = new ArrayList<Variable>();
    private List<Variable> varCols = new ArrayList<Variable>();

    /**
     * Generate an indicator
     * @param tblKeys
     * @param columns
     * @param rows
     * @param valueindex
     * @return
     */
    public DataTableIndicator generate(DataTable tblKeys, List<Variable> columns, List<Variable> rows, int valueindex) {
        DataTableIndicator tbl = new DataTableIndicatorImpl();

        for (Row sourceRow: tblKeys.getRows()) {
            Long val = (Long)sourceRow.getValue(valueindex);

            List<Key> keys = (List<Key>)sourceRow.getValues();

            IndicatorColumn col = resolveColumn(tbl, columns, keys);
            IndicatorRow row = resolveRow(tbl, rows, keys);

            tbl.setIndicatorValue(col, row, val.doubleValue());
        }

        return tbl;
    }

    private IndicatorRow resolveRow(DataTableIndicator tbl, List<Variable> vars, List<Key> keys) {
        IndicatorRow parent = null;

        for (Variable var: vars) {
            Key key = findKeyByVariable(keys, var);

            if (var.isGrouped()) {
                IndicatorRow row = tbl.findRowByKey(key.getGroup(), parent);
                if (row == null) {
                    row = tbl.addIndicatorRow(parent);
                    row.setTitle(var.getGroupDisplayText(key));
                    row.setKey(key.getGroup());
                }
                parent = row;
            }

            IndicatorRow row = tbl.findRowByKey(key.getValue(), parent);
            if (row == null) {
                row = tbl.addIndicatorRow(parent);
                row.setTitle(var.getDisplayText(key));
                row.setKey(key.getValue());
            }
            parent = row;
        }
        return parent;
    }

    private IndicatorColumn resolveColumn(DataTableIndicator tbl, List<Variable> vars, List<Key> keys) {
        IndicatorColumn parent = null;

        for (Variable var: vars) {
            Key key = findKeyByVariable(keys, var);

            if (var.isGrouped() && key.getGroup() != null) {
                IndicatorColumn row = tbl.findColumnByKey(key.getGroup(), parent);
                if (row == null) {
                    row = tbl.addIndicatorColumn(parent);
                    row.setTitle(var.getGroupDisplayText(key));
                    row.setKey(key.getGroup());
                }
                parent = row;
            }

            IndicatorColumn row = tbl.findColumnByKey(key.getValue(), parent);
            if (row == null) {
                row = tbl.addIndicatorColumn(parent);
                row.setTitle(var.getDisplayText(key));
                row.setKey(key.getValue());
            }
            parent = row;
        }
        return parent;
    }

    private Object[] createKeys(List<Variable> vars, List<Key> keys) {
        List vals = new ArrayList();

        for (Variable var: vars) {
            Key key = findKeyByVariable(keys, var);
            if (key.getGroup() != null) {
                vals.add(key.getGroup());
            }
            vals.add(key.getValue());
        }
        return vals.toArray();
    }

    private Key findKeyByVariable(List<Key> keys, Variable var) {
        for (Key key: keys) {
            if (key.getVariable() == var) {
                return key;
            }
        }

        throw new RuntimeException("No key found for variable " + var.getId());
    }

}
